/**
* \file: AlsaAudioOutImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <memory.h>
#include <pthread.h>
#include <sys/prctl.h>
#include <limits.h>
#include <adit_logging.h>
#include <dipo_macros.h>
#include "AlsaAudioOut.h"
#include "AlsaAudioOutImpl.h"
#include "AlsaConfiguration.h"

#define MODE_RTS 7
#define MODE_ALSA 8
#define MODE MODE_ALSA

using namespace std;

LOG_IMPORT_CONTEXT(cply);

namespace adit { namespace carplay
{

static const char* _getChannelName(AudioChannelType inChannel)
{
    if (inChannel == AudioChannelType_Main)
        return "main";
    else if (inChannel == AudioChannelType_Alternate)
        return "alternate";
    else
        return "";
}

#if (MODE == MODE_RTS)
static void _splitBuffers16(void** outDest, void* inSrc, uint32_t inCount)
{
    uint8_t** dests = reinterpret_cast<uint8_t**>(outDest);
    uint8_t* dest[2] = { dests[0], dests[1] };
    uint8_t* src = reinterpret_cast<uint8_t*>(inSrc);

    for (uint32_t i = 0; i < inCount; i++)
    {
        *(dest[0]) = *src; dest[0]++; src++; // 1st 16-bit
        *(dest[0]) = *src; dest[0]++; src++;

        *(dest[1]) = *src; dest[1]++; src++; // 2nd 16-bit
        *(dest[1]) = *src; dest[1]++; src++;
    }
}

static void _splitBuffers24(void** outDest, void* inSrc, uint32_t inCount)
{
    uint8_t** dests = reinterpret_cast<uint8_t**>(outDest);
    uint8_t* dest[2] = { dests[0], dests[1] };
    uint8_t* src = reinterpret_cast<uint8_t*>(inSrc);

    for (uint32_t i = 0; i < inCount; i++)
    {
        *dest[0] = *src; dest[0]++; src++; // 1st 24-bit
        *dest[0] = *src; dest[0]++; src++;
        *dest[0] = *src; dest[0]++; src++;

        *dest[1] = *src; dest[1]++; src++; // 2nd 24-bit
        *dest[1] = *src; dest[1]++; src++;
        *dest[1] = *src; dest[1]++; src++;
    }
}
#endif

AlsaAudioOut::Impl::Impl()
{
    config = nullptr;
    source = nullptr;

    processThreadId = 0;
#if (MODE == MODE_RTS)
    handle = 0;
#endif

    periodMilli = 0;
    periodSamples = 0;
    appendEmptyPeriods = 0;
    channel = AudioChannelType_Main;

    running = false;
    flush = false;
}

AlsaAudioOut::Impl::~Impl()
{
    /* PRQA: Lint Message 1506: Stop is not expected to be further overridden */
    /*lint -save -e1506*/
    Stop();
    /*lint -restore*/
}

bool AlsaAudioOut::Impl::Initialize(const IConfiguration& inConfig,
        IAudioOutSource& inSource)
{
    config = &inConfig;
    source = &inSource;

    bool verboseLogging = AlsaConfiguration::Instance().IsVerboseLogging(inConfig);
    bool alsaLogging = AlsaConfiguration::Instance().IsAlsaLogging(inConfig);

    alsa = move(unique_ptr<AlsaAudioCommon>(new AlsaAudioCommon(alsaLogging, verboseLogging, inConfig)));
    return true;
}

bool AlsaAudioOut::Impl::Prepare(AudioFormatStruct inFormat, AudioChannelType inChannel,
        const std::string& inAudioType)
{
	int32_t buffer_periods = -1;
	int32_t silence_ms = -1;
	int32_t inittout_ms = -1;

	(void)inFormat;

    if (config == nullptr)
    {
        LOG_ERROR((cply, "AlsaAudioOut is not initialized"));
        return false;
    }

    channel = inChannel;
    format = inFormat;

    if (format.BitsPerChannel != 24 && format.BitsPerChannel != 16)
    {
        LOG_ERROR((cply, "AlsaAudioOut does not support %d bit audio", format.BitsPerChannel));
        return false;
    }

    // numbers of periods to append after end-of-stream
    appendEmptyPeriods = config->GetNumber("alsa-audio-out-append-empty-periods", 0);

    if (!AlsaConfiguration::Instance().GetDeviceSettings(*config,
            (AlsaConfiguration::Channel) inChannel, inAudioType, inFormat, deviceName /* out */,
            periodMilli /* out */, buffer_periods /* out */, silence_ms /* out */, inittout_ms /* out */))
    {
        LOG_ERROR((cply, "failed to get audio device settings"));
        return false; /* ========== leaving function ========== */
    }

    // get period size in ms, could be dynamically set
#if 0
    periodMilli = getPeriodMilli(deviceName, format.SampleRate);
#endif

#if (MODE == MODE_RTS)
    periodSamples = periodMilli * format.SampleRate / 1000;
#endif

#if (MODE == MODE_ALSA)
    alsa->SetDir( SND_PCM_STREAM_PLAYBACK );
    alsa->SetRate( format.SampleRate );
    alsa->SetChannels( format.Channels );
    alsa->SetIdealPeriodMs( periodMilli );
    if (buffer_periods >= 0)
        alsa->SetBufferIdealPeriods( buffer_periods );
    if (silence_ms >= 0)
        alsa->SetPrefillMs ( silence_ms );
    if (inittout_ms >= 0)
        alsa->SetInitTout ( inittout_ms );

    switch (format.BitsPerChannel) {
        case 16 : {
            alsa->SetFormat( SND_PCM_FORMAT_S16_LE );
            break;
        }
        case 24 : {
            alsa->SetFormat( SND_PCM_FORMAT_S24_3LE );/* TODO check if this is correct 24bit data in 3bytes or if S24_LE 24bit in 4bytes is correct */
            break;
        }
        default :
            LOG_ERROR((cply, "AlsaAudioOut does not support %d bits per channel\n", format.BitsPerChannel));
            return false;
    }

    if (0 != alsa->SetupPCM((char*)deviceName.c_str())) {
        return false;
    }

    periodSamples = alsa->GetIdealPeriodMs();
#endif

    LOGD_DEBUG((cply, "%s audio out: device=%s, period=%dms", _getChannelName(channel),
            deviceName.c_str(), periodMilli));

#if (MODE == MODE_RTS)
    memset(&alsaDevice, 0, sizeof(alsaDevice));
    alsaDevice.period_frames = periodSamples;
    alsaDevice.rate = format.SampleRate;
    alsaDevice.format = SND_PCM_FORMAT_S16_LE; // TODO
    alsaDevice.pcmname = deviceName.c_str();
    alsaDevice.dir = SND_PCM_STREAM_PLAYBACK;

    memset(&alsaStreams, 0, sizeof(alsaStreams));
    alsaStreams[0].adevidx = 0;
    alsaStreams[0].channel = 0;
    alsaStreams[1].adevidx = 0;
    alsaStreams[1].channel = 1;

    memset(&alsaConfig, 0, sizeof(alsaConfig));
    alsaConfig.num_adevs = 1;
    alsaConfig.adevs = &alsaDevice;
    alsaConfig.num_streams = format.Channels;
    alsaConfig.streams = alsaStreams;
    alsaConfig.prefill_ms = periodMilli * 2; // two periods
    alsaConfig.features = RTS_FEAT_FORCE_CFG_RELOAD;
#endif

    return true;
}

bool AlsaAudioOut::Impl::Start()
{
    running = true;

    if (0 != pthread_create(&processThreadId, nullptr, process, this))
    {
        LOG_ERROR((cply, "could not create audio out thread"));
        return false;
    }

    LOGD_DEBUG((cply, "%s audio out started", _getChannelName(channel)));
    return true;
}

void AlsaAudioOut::Impl::Stop()
{
    bool wasRunning = running;
    running = false;

    /* Abort streaming to avoid recover retries (long blocking) */
    alsa->AbortStreaming();

    if (processThreadId != 0)
    {
        pthread_join(processThreadId, nullptr);
        processThreadId = 0;
    }

    if (wasRunning)
        LOGD_DEBUG((cply, "%s audio out stopped", _getChannelName(channel)));
}

void AlsaAudioOut::Impl::Flush()
{
    flush = true;
}

void* AlsaAudioOut::Impl::process(void* inData)
{
    (void)inData;

    auto me = static_cast<AlsaAudioOut::Impl*>(inData);
    dipo_return_value_on_invalid_argument(cply, me == nullptr, nullptr);

    // set thread name
    char buf[16];
    snprintf(buf, sizeof(buf), "Alsa%sAudioOut",
            (me->channel == AudioChannelType_Main ? "Main" : "Alt"));
    prctl(PR_SET_NAME, buf, 0, 0, 0);

    // set thread prioirty
    me->alsa->SetThreadPriority((const char*)&buf[0]);

    // create RTS
#if (MODE == MODE_RTS)
    int err = rts_create(&me->handle, &me->alsaConfig);
    if (err < 0)
    {
        LOG_ERROR((cply, "create RTS streaming returned with error %d", err));
        return nullptr;
    }
#endif

    uint64_t sampleNumber = 0;

    while (me->running)
    {
        const int bufferSize = me->periodSamples *
                (me->format.BitsPerChannel / 8) * me->format.Channels;
        char buffer[bufferSize];

        Samples samples;
        samples.DataPtr = buffer;
        samples.Length = sizeof(buffer);

#if 0
        if (me->audioSink != nullptr)
        {
            gint64 position = 0;
            GstFormat format = GST_FORMAT_DEFAULT;
            gst_element_query_position(me->audioSink, &format, &position);

            samples.TimeStamp = position;
        }
        else
#endif
        {
            // default fallback to samples to were pushed to the pipeline
            samples.TimeStamp = sampleNumber;
        }

        // returns immediately if not RTP packet arrived yet and before 10ms after starting
        // samples are zeroed
        me->source->Read(samples);
        sampleNumber += me->periodSamples;

#if (MODE == MODE_RTS)
        void* buffers[2] = { buffer, nullptr };

        char bufferLeft[bufferSize / 2];
        char bufferRight[bufferSize / 2];

        if (me->format.Channels == 2)
        {
            buffers[0] = bufferLeft;
            buffers[1] = bufferRight;

            if (me->format.BitsPerChannel == 24)
                _splitBuffers24(buffers, buffer, me->periodSamples);
            else
                _splitBuffers16(buffers, buffer, me->periodSamples);
        }

        // blocks when buffer queue is full
        int err = 0;
        if ((err = rts_write(me->handle, buffers)) < 0)
        {
            LOG_ERROR((cply, "write to RTS failed with %d", err));
            break;
        }
#endif
#if (MODE == MODE_ALSA)
        int err = 0;
        if(me->flush){
            /*
             * Flush operation is implemented with snd_pcm_drop which
             * discards the complete amount of data and then timestamp is
             * declared at a "wrong" point of time for which _UpdateEstimatedRate
             * will report a sample rate not matching with the expected.
             * Buffers kept by ALSA are so small that the user impact of not
             * flushing can be neglected.
             * Therefore, not flushing at all is current solution.
             *
             * me->alsa->FlushBuffers();
             */
            me->flush = false;
        }
        if ((err = me->alsa->ReadWrite(buffer, bufferSize)) < 0)
        {
            LOG_ERROR((cply, "write to RTS failed with %d", err));
            break;
        }
#endif
    }

#if (MODE == MODE_RTS)
    if (me->appendEmptyPeriods > 0)
    {
        char emptyBuffer[me->periodSamples * me->format.BitsPerChannel / 8];
        memset(emptyBuffer, 0, sizeof(emptyBuffer));

        const int max = me->appendEmptyPeriods;
        for (int i = 0; i < max; i++)
        {
            char* buffers[] = { emptyBuffer, emptyBuffer };

            // blocks when buffer queue is full
            int err = 0;
            if ((err = rts_write(me->handle, (void**)buffers)) < 0)
            {
                LOG_ERROR((cply, "write to RTS failed with %d", err));
                break;
            }
        }
    }
#endif

#if (MODE == MODE_ALSA)
    me->alsa->StopStreaming();
    me->alsa->ReleasePCM();
#endif

#if (MODE == MODE_RTS)
    if (me->handle != 0)
    {
        int err = rts_destroy(me->handle);
        if (err < 0)
        {
            LOG_ERROR((cply, "error on RTS destroy with %d", err));
        }
    }
#endif

    return nullptr;
}

} } // namespace adit { namespace carplay
